<?php

namespace App\Helpers;

use App\Http\Controllers\MailController;
use App\Http\Controllers\WhatsAppController;
use App\Mail\buddhaBundle\OrderConfirmationMail;
use App\Mail\instaReels\OrderConfirmationMail as InstaReelsOrderConfirmationMail;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MetaHelper
{
    public static function setCookie($txnId, $productKey)
    {
        // Create a unique cookie name for each product key
        $cookieName = $productKey . '_' . $txnId;
        setcookie($cookieName, $txnId, time() + (30 * 24 * 60 * 60), '/');
    }

    public static function getCookie($productKey)
    {
        $prefix = $productKey . '_';
        $latestValue = null;
        $latestKeyValue = 0;

        // Loop through all cookies and find those that match the prefix.
        foreach ($_COOKIE as $key => $value) {
            if (strpos($key, $prefix) === 0) {
                // Extract the part after the underscore.
                $txnPart = substr($key, strlen($prefix));

                // If it's numeric, we compare it.
                if (is_numeric($txnPart)) {
                    $currentNumeric = intval($txnPart);
                    if ($currentNumeric > $latestKeyValue) {
                        $latestKeyValue = $currentNumeric;
                        $latestValue = $value;
                    }
                } else {
                    // If not numeric, you could simply override the value or implement another comparison.
                    $latestValue = $value;
                }
            }
        }

        return $latestValue;
    }



    public static function markTransactionSuccess($transaction, $test_event_code = null, $renotify = false)
    {
        if ($renotify) {
            Log::info('Renotify: ' . $transaction->client_txn_id);
        }

        $product_key = $transaction->product_key;
        $product = ProductsHelper::getProduct($product_key);

        $transaction->status = 'success';
        $transaction->save();
        Log::info('Transaction marked as success: ' . $transaction->client_txn_id);

        if ($transaction->platform == 'meta') {
            self::sendPurchaseEventToMeta($transaction, $test_event_code, $product);
        }
        if ($transaction->platform == 'google') {
            self::sendPurchaseEventToGoogleAds($transaction, $product);
        }
        if ($renotify) {
            return;
        }
        self::sendMail($transaction, $product);
        WhatsAppController::sendSucessWhatsapp($transaction);
    }

    private static function sendMail($transaction, $product)
    {
        $subject = $transaction->user_name . ', Your Order Has Been Completed! ✅';
        $htmlContent = MailController::renderMailableToHtml($transaction, 'success');
        MailController::sendZohoMail($transaction, $subject, $htmlContent);

        // Return Email Preview
        if ($product['key'] == 'buddha-bundle') {
            return new OrderConfirmationMail($transaction, 'success');
        } elseif ($product['key'] == 'insta-reels') {
            return new InstaReelsOrderConfirmationMail($transaction, 'success');
        }
    }

    //api center nhi mila toh comment out kr diya, google ads me.
    // private static function sendPurchaseEventToGoogleAds($transaction, $product)
    // {
    //     $client = new Google\Client();
    //     $client->setClientId('YOUR_CLIENT_ID');
    //     $client->setClientSecret('YOUR_CLIENT_SECRET');
    //     $client->refreshToken('YOUR_REFRESH_TOKEN');

    //     $service = new Google\Service\GoogleAds($client);

    //     $conversion = new Google\Service\GoogleAds\GoogleAdsConversion([
    //         'conversion_action' => "customers/{customerId}/conversionActions/{$product['google_conversion_id']}",
    //         'conversion_date_time' => now()->format('Y-m-d H:i:s'),
    //         'conversion_value' => $product['price'],
    //         'currency_code' => 'INR',
    //         'order_id' => $transaction->client_txn_id,
    //         'user_identifiers' => [
    //             new Google\Service\GoogleAds\GoogleAdsUserIdentifier([
    //                 'hashed_email' => hash('sha256', strtolower(trim($transaction->user_email)))
    //             ]),
    //             new Google\Service\GoogleAds\GoogleAdsUserIdentifier([
    //                 'hashed_phone_number' => hash('sha256', preg_replace('/\D/', '', "91" . $transaction->user_mobile))
    //             ])
    //         ],
    //         'user_agent' => $transaction->user_agent,
    //         'ip_address' => $transaction->ip_address
    //     ]);

    //     $response = $service->conversions->upload([$conversion]);
    //     Log::info('Google Ads API Response: ' . json_encode($response));
    // }

    private static function sendPurchaseEventToGoogleAds($transaction, $product)
    {
        Log::info('Sending purchase event: ' . $transaction->client_txn_id);

        $phone = '+91' . preg_replace('/\D/', '', $transaction->user_mobile);
        $url = "https://www.googleadservices.com/pagead/conversion/awcid={$product['google_aw_id']}/conversiononly.gif";
        // $url .= "?test=1"; // Add this for testing

        $params = [
            'value' => $product['price'], // Changed from 'amt'
            'currency' => 'INR', // Changed from 'cur'
            'oid' => $transaction->client_txn_id,
            'email' => hash('sha256', strtolower(trim($transaction->user_email))),
            'phone_number' => hash('sha256', $phone), // Fixed parameter name + hashing
            'uip' => $transaction->ip_address,
            'ua' => $transaction->user_agent,
            'label' => $product['google_purchase_conversion_label'],
            'random' => time(),
            // Optional consent parameters:
            'ad_user_data_consent' => 'GRANTED',
            'ad_personalization_consent' => 'GRANTED'
        ];

        Log::info('Params: ' . json_encode($params));

        try {
            $response = Http::get($url, $params);
            Log::info('Response Body: ' . $response->body()); // Will show 1x1 pixel if successful
            return ['status' => $response->status()];
        } catch (\Exception $e) {
            Log::error('Error: ' . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }

    private static function sendPurchaseEventToMeta($transaction, $test_event_code = null, $product)
    {

        $event_id = $transaction->client_txn_id;
        $email = $transaction->user_email;
        $phone = "91" . $transaction->user_mobile;
        $fbc = $transaction->fbc;
        $ip_address = $transaction->ip_address;
        $user_agent = $transaction->user_agent;

        Log::info('Sending purchase event to Meta: ' . $event_id);
        $pixel_id = $product['pixel_id'];
        $access_token = $product['meta_access_token'];

        $url = "https://graph.facebook.com/v18.0/{$pixel_id}/events";

        $user_data = [
            'em' => [hash('sha256', strtolower(trim($email)))],
            'ph' => [hash('sha256', preg_replace('/\D/', '', $phone))],
        ];

        if (!empty($fbc)) {
            $user_data['fbc'] = $fbc;
        }
        if (!empty($ip_address)) {
            $user_data['client_ip_address'] = $ip_address;
        }
        if (!empty($user_agent)) {
            $user_data['client_user_agent'] = $user_agent;
        }

        $payload = [
            'data' => [
                [
                    'event_name' => 'Purchase',
                    'event_time' => time(),
                    'event_id' => $event_id,
                    'user_data' => $user_data,
                    'custom_data' => [
                        'value' => $product['price'],
                        'currency' => 'INR',
                    ],
                    'action_source' => 'website',
                ]
            ],
            'access_token' => $access_token
        ];
        // Add test_event_code if provided
        if ($test_event_code) {
            $payload['test_event_code'] = $test_event_code;
        }

        //log payload
        Log::info('Payload: ' . json_encode($payload));

        $response = Http::post($url, $payload);

        return $response->json();
    }
}
