<?php

namespace App\Http\Controllers;

use App\Helpers\MetaHelper;
use App\Helpers\ProductsHelper;
use App\Mail\buddhaBundle\OrderCancelledMail;
use App\Mail\buddhaBundle\OrderConfirmationMail;
use App\Mail\instaReels\OrderCancelledMail as InstaReelsOrderCancelledMail;
use App\Mail\instaReels\OrderConfirmationMail as InstaReelsOrderConfirmationMail;
use App\Models\Settings;
use App\Models\Transaction;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Database\Events\TransactionBeginning;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\View;
use Twilio\TwiML\Voice\Transcription;

class MailController extends Controller
{

    public function unsubscribe($txnId)
    {
        $logMessage = "Unsubscribe URL clicked txnId: " . $txnId . " at " . now()->format('Y-m-d H:i:s') . "\n"; // Log with timestamp

        $transaction = Transaction::where('client_txn_id', $txnId)->first();
        if (!$transaction) {
            //return back
            return back();
        }

        // Append log to the file
        file_put_contents(public_path('reorder.txt'), $logMessage, FILE_APPEND);

        $product = ProductsHelper::getProduct($transaction->product_key);
        $app_link = $product['app_link'];

        // Append log to the file
        file_put_contents(public_path('reorder.txt'), $logMessage, FILE_APPEND);

        //redirect to app link
        return redirect($app_link);
    }

    public function reorder($txnId)
    {
        $logMessage = "Reorder URL clicked ID: " . $txnId . " at " . now()->format('Y-m-d H:i:s') . "\n"; // Log with timestamp

        $transaction = Transaction::where('client_txn_id', $txnId)->first();
        if (!$transaction) {
            //return back
            return back();
        }

        $product = ProductsHelper::getProduct($transaction->product_key);
        $app_link = $product['app_link'];

        // Append log to the file
        file_put_contents(public_path('reorder.txt'), $logMessage, FILE_APPEND);

        //redirect to app link
        return redirect($app_link);
    }

    public function openOrder($txnId)
    {
        $transaction = Transaction::where('client_txn_id', $txnId)->first();
        if (!$transaction) {
            abort(404, 'Transaction not found');
        }
        return self::returnMailClass($transaction, 'success');
    }

    public function openMailFree($product)
    {
        $transaction = Transaction::where('user_mobile', '1234567890')
            ->orWhere('user_mobile', '8860554455')
            ->orderBy('id', 'desc')
            ->where('product_key', $product)
            ->first();

        if (!$transaction) {
            abort(404, 'No pending transactions found!');
        }
        return self::returnMailClass($transaction, 'success');
    }

    private static function returnMailClass($transaction, $type)
    {
        $product = ProductsHelper::getProduct($transaction->product_key);
        if (!$product || !isset($product['key'])) {
            return NULL;
        }
        if ($product['key'] == 'insta-reels') {
            if ($type == 'cancel') {
                return new InstaReelsOrderCancelledMail($transaction);
            } else {
                return new InstaReelsOrderConfirmationMail($transaction);
            }
        } elseif ($product['key'] == 'buddha-bundle') {
            if ($type == 'cancel') {
                return new OrderCancelledMail($transaction);
            } else {
                return new OrderConfirmationMail($transaction);
            }
        } elseif ($product['key'] == 'buddha-bundle-bg') {
            if ($type == 'cancel') {
                return new OrderCancelledMail($transaction);
            } else {
                return new OrderConfirmationMail($transaction);
            }
        } else {
            return NULL;
        }
    }

    function fetchZohoAccountId()
    {
        $client = new Client();

        try {
            $token = self::refreshZohoAccessToken();
            $response = $client->get('https://mail.zoho.in/api/accounts', [
                'headers' => [
                    'Authorization' => "Zoho-oauthtoken " . $token,
                    'Content-Type' => 'application/json',
                ],
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $account) {
                    Log::info('Account ID: ' . $account['accountId']);
                    Log::info('Email Address: ' . $account['emailAddress']);
                }
                return $data['data'][0]['accountId'];
            } else {
                Log::error('Failed to retrieve account details', $data);
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Zoho Mail API error: ' . $e->getMessage());
            return null;
        }
    }



    public static function sendZohoMail($transaction, $subject, $htmlContent)
    {
        $product = ProductsHelper::getProduct($transaction->product_key);
        $mail_from_name = $product['mail_from_name'];
        $toAddress = $transaction->user_email;
        $token = self::refreshZohoAccessToken();
        $client = new Client();

        $accountId = env('ZOHO_ACCOUNT_ID'); // Replace with your actual account ID
        $fromAddress = $mail_from_name . " <" . env('MAIL_FROM_ADDRESS') . ">";

        try {
            $response = $client->post("https://mail.zoho.in/api/accounts/{$accountId}/messages", [
                'headers' => [
                    'Authorization' => "Zoho-oauthtoken {$token}",
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'fromAddress' => $fromAddress,
                    'toAddress' => $toAddress,
                    'subject' => $subject,
                    'content' => $htmlContent,
                    'mailFormat' => 'html',
                ],
            ]);

            $data = json_decode($response->getBody(), true);
            if (isset($data['data']['messageId'])) {
                Log::info('Email sent successfully. Message ID: ' . $data['data']['messageId']);
                return $data['data']['messageId'];
            } else {
                Log::error('Failed to send email. Response: ' . json_encode($data));
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Zoho Mail API error: ' . $e->getMessage());
            return null;
        }
    }



    public function test()
    {
        // return;

        $transaction = Transaction::where('user_mobile', '8860554455')->orderBy('id', 'desc')->first();
        $product = ProductsHelper::getProduct($transaction->product_key);


        // if (!$transaction) {
        //     return "No pending transactions found!";
        // }
        // return new OrderCancelledMail($transaction);

        // MetaHelper::markTransactionSuccess($transaction);
        // return 'sent';

        $subject = $transaction->user_name . ', Your Order Has Been Completed! ✅';
        $htmlContent = self::renderMailableToHtml($transaction, 'success');
        self::sendZohoMail($transaction, $subject, $htmlContent);

        return self::returnMailClass($transaction, 'success');
    }

    public static function renderMailableToHtml($transaction, $type = 'cancel')
    {
        $mailable = self::returnMailClass($transaction, $type);
        return $mailable->render();
    }

    function generateZohoAccessToken()
    {
        $client = new Client();
        $authCode = env('ZOHO_AUTH_CODE'); // Replace with the actual code

        try {
            $response = $client->post('https://accounts.zoho.in/oauth/v2/token', [
                'form_params' => [
                    'grant_type' => 'authorization_code',
                    'client_id' => env('ZOHO_CLIENT_ID'),
                    'client_secret' => env('ZOHO_CLIENT_SECRET'),
                    'redirect_uri' => env('ZOHO_REDIRECT_URL'),
                    'code' => $authCode,
                    'access_type' => 'offline',
                ],
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['access_token']) && isset($data['refresh_token'])) {
                Log::info('New Zoho Access Token: ' . $data['access_token']);
                Log::info('New Zoho Refresh Token: ' . $data['refresh_token']);

                // Save these tokens in the .env file manually or use a database
                return [
                    'access_token' => $data['access_token'],
                    'refresh_token' => $data['refresh_token'],
                ];
            } else {
                Log::error('Failed to generate Zoho tokens', $data);
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Zoho API error: ' . $e->getMessage());
            return null;
        }
    }


    private static function refreshZohoAccessToken()
    {
        $settings = Settings::first();
        if ($settings && $settings->access_token_zoho && $settings->access_token_updated_at->diffInMinutes(Carbon::now()) < 5) {
            return $settings->access_token_zoho;
        }

        $client = new Client();
        $refreshToken = env('ZOHO_REFRESH_TOKEN');
        if (!$refreshToken) {
            Log::error('Zoho refresh token not found');
            return null;
        }

        try {
            $response = $client->post('https://accounts.zoho.in/oauth/v2/token', [
                'form_params' => [
                    'grant_type' => 'refresh_token',
                    'client_id' => env('ZOHO_CLIENT_ID'),
                    'client_secret' => env('ZOHO_CLIENT_SECRET'),
                    'refresh_token' => $refreshToken,
                ],
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['access_token'])) {
                $settings->access_token_zoho = $data['access_token'];
                $settings->access_token_updated_at = Carbon::now();
                $settings->save();
                Log::info('New Zoho Access Token: ' . $data['access_token']);

                return $data['access_token'];
            } else {
                Log::error('Failed to refresh Zoho token', $data);
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Zoho token refresh error: ' . $e->getMessage());
            return null;
        }
    }

    // Send email to users after payment failure
    public function sendMail()
    {

        Log::info('Sending email to users after payment failure');
        $transactions = Transaction::where('status', 'pending')
            ->where('created_at', '<=', Carbon::now()->subMinutes(10))
            ->whereNull('first_email_sent_at') // Ensures email is not sent again
            ->whereNotNull('user_email') // Ensure email exists
            ->get();

        foreach ($transactions as $transaction) {
            try {
                $subject = $transaction->user_name . ', Your Order Has Been Cancelled! ❌';
                $htmlContent = $this->renderMailableToHtml($transaction, 'cancel');
                $this->sendZohoMail($transaction, $subject, $htmlContent);

                // Mark first email as sent
                $transaction->first_email_sent_at = Carbon::now();
                $transaction->save();
            } catch (\Throwable $th) {
                //throw $th;
                //log $th->message
                Log::error("Failed to Send Email to " . $transaction->user_email . " Error: " . $th->getMessage());
            }
        }

        return "Cancellation emails sent!";
    }

    // Send second reminder email at 9 AM next day
    public function sendReminderMail()
    {
        Log::info('Sending reminder email to users after payment failure');
        $transactions = Transaction::where('status', 'pending')
            ->where('created_at', '<=', Carbon::now()->subDay()->startOfDay()->addHours(9))
            ->whereNotNull('first_email_sent_at') // Ensure first email was sent
            ->whereNull('second_email_sent_at') // Ensures second email is not sent again
            ->whereNotNull('user_email') // Ensure email exists
            ->get();

        foreach ($transactions as $transaction) {
            try {
                $subject = $transaction->user_name . ', Your Order Has Been Cancelled! ❌';
                $htmlContent = $this->renderMailableToHtml($transaction, 'cancel');
                $this->sendZohoMail($transaction, $subject, $htmlContent);

                // Mark second email as sent
                $transaction->second_email_sent_at = Carbon::now();
                $transaction->save();
            } catch (\Throwable $th) {
                //log which user failed to receive email
                Log::error("Failed to Send Email to " . $transaction->user_email . " Error: " . $th->getMessage());
            }
        }

        return "Reminder emails sent!";
    }
}
